INFORMANT_VERSION = "3.9.0.1003"
local getItem, addLine, clear, frameActive, frameLoaded, getCatName, getFilter, getFilterVal, getItem, getRowCount, nilSafeString, onEvent, onLoad, onVariablesLoaded, onQuit, scrollUpdate, setDatabase, setFilter, setFilterDefaults, showHideInfo, split, tooltipHandler, getKeyBindProfile, whitespace

-- LOCAL VARIABLES

local self = {}
local lines = {}
local itemInfo = nil

-- GLOBAL VARIABLES

BINDING_HEADER_INFORMANT_HEADER = _INFM('BindingHeader')
BINDING_NAME_INFORMANT_POPUPDOWN = _INFM('BindingTitle')

InformantConfig = {}

-- LOCAL DEFINES

local filterDefaults = {
		['all'] = 'on',
		['embed'] = 'off',
		['locale'] = 'default',
		['show-vendor'] = 'on',
		['show-vendor-sell'] = 'on',
		['show-usage'] = 'on',
		['show-stack'] = 'on',
		['show-icon'] = 'on',
		['show-id'] = 'on',
	}

-- FUNCTION DEFINITIONS

function split(str, at)
	local splut = {}

	if (type(str) ~= "string") then return nil end
	if (not str) then str = "" end
	if (not at)
		then table.insert(splut, str)
	else
		for n, c in string.gfind(str, '([^%'..at..']*)(%'..at..'?)') do
			table.insert(splut, n)
			if (c == '') then break end
		end
	end
	return splut
end

function getItem(itemID)
	local baseData = self.database[itemID]
	if (not baseData) then 
		return getItemBasic(itemID)
	end

	local _, _, _, iLevel, sType, _, iCount, _, sTexture = GetItemInfo(itemID)
	local baseSplit = split(baseData, ",")
	local sell = tonumber(baseSplit[1])
	local stack = tonumber(iCount)
	
	local dataItem = {
		['sell'] = sell,
		['stack'] = stack,
		['texture'] = sTexture,
		['fullData'] = true,
	}
	if (sType) then
		dataItem.classText = sType
	else
		dataItem.classText = "Unknown"
	end

	return dataItem
end

function getItemBasic(itemID)
	if (not itemID) then return end
	local sName, sLink, iQuality, iLevel, sType, sSubType, iCount, sEquipLoc, sTexture = GetItemInfo(tonumber(itemID))

	if (sName) then
		local dataItem = {
			['classText'] = sType,
			['quality'] = iQuality,
			['stack'] = iCount,
			['texture'] = sTexture,
			['reqLevel'] = iLevel,
			['fullData'] = false,
		}
	return dataItem
	end
end

function setDatabase(database)
	self.database = database
	Informant.SetDatabase = nil -- Set only once
end

function setFilter(key, value)
	if (not InformantConfig.filters) then
		InformantConfig.filters = {};
		setFilterDefaults()
	end
	if (type(value) == "boolean") then
		if (value) then
			InformantConfig.filters[key] = 'on';
		else
			InformantConfig.filters[key] = 'off';
		end
	else
		InformantConfig.filters[key] = value;
	end
end

function getFilterVal(type)
	if (not InformantConfig.filters) then
		InformantConfig.filters = {}
		setFilterDefaults()
	end
	return InformantConfig.filters[type]
end

function getFilter(filter)
	value = getFilterVal(filter)
	if ((value == _INFM('CmdOn')) or (value == "on")) then return true
	elseif ((value == _INFM('CmdOff')) or (value == "off")) then return false end
	return true
end

function getLocale()
	local locale = Informant.GetFilterVal('locale');
	if (locale ~= 'on') and (locale ~= 'off') and (locale ~= 'default') then
		return locale;
	end
	return GetLocale();
end

local categories
function getCatName(catID)
	if (not categories) then categories = {GetAuctionItemClasses()} end
	for cat, name in categories do
		if (cat == catID) then return name end
	end
end

function tooltipHandler(funcVars, retVal, frame, name, link, quality, count, price)
	-- nothing to do, if informant is disabled
	if (not getFilter('all')) then
		return;
	end;

	if EnhTooltip.LinkType(link) ~= "item" then return end

	local sell = 0
	local stacks = 1

	local itemID, randomProp, enchant, uniqID, lame = EnhTooltip.BreakLink(link)
	
	if (itemID and itemID > 0) and (Informant) then
		itemInfo = getItem(itemID)
	end
	if (not itemInfo) then return end

	itemInfo.itemName = name
	itemInfo.itemLink = link
	itemInfo.itemCount = count
	itemInfo.itemID=itemID

	stacks = itemInfo.stack
	if (not stacks) then stacks = 1 end

	sell = tonumber(itemInfo.sell) or 0

	itemInfo.itemSell = sell

	local embedded = getFilter('embed')

	if (getFilter('show-icon')) then
		if (itemInfo.texture) then
			EnhTooltip.SetIcon(itemInfo.texture)
		end
	end

	if (getFilter('show-vendor')) then
		if (sell > 0) then
			local sgsc = EnhTooltip.GetTextGSC(sell, true)

			if (getFilter('show-vendor-sell')) then
				if (count and (count > 1)) then
					EnhTooltip.AddLine(string.format(_INFM('FrmtInfoSellmult'), sgsc), (sell*count), embedded, true)
				else
					EnhTooltip.AddLine(string.format(_INFM('FrmtInfoSell')), sell, embedded, true)
				end
			end
		end
	end
	if (getFilter('show-stack')) then
		if (stacks > 1) then
			EnhTooltip.AddLine(string.format(_INFM('FrmtInfoStx'), stacks), nil, embedded)
		end
	end
	
	if (getFilter('show-usage')) then
		local reagentInfo = ""
		if (itemInfo.classText) then
			reagentInfo = string.format(_INFM('FrmtInfoClass'), itemInfo.classText)
			EnhTooltip.AddLine(reagentInfo, nil, embedded)
			EnhTooltip.LineColor(0.6, 0.4, 0.8)
		end
	end

	if (getFilter('show-id')) then
		if itemInfo.itemID then
			EnhTooltip.AddLine(string.format(_INFM('FrmtInfoID'), itemInfo.itemID), nil, embedded)
			EnhTooltip.LineColor(0.8, 0.5, 0.6)
		end
	end
end

function nilSafeString(str)
	if (not str) then str = "" end
	return str;
end

function whitespace(length)
	local spaces = ""
	while length ~= 0 do
		spaces = spaces.." "
		length = length - 1
	end
	return spaces
end

function showHideInfo()
	if (InformantFrame:IsVisible()) then
		InformantFrame:Hide()
	elseif (itemInfo) then
		InformantFrameTitle:SetText(_INFM('FrameTitle'))

		-- Woohoo! We need to provide any information we can from the item currently in itemInfo

		local color = "ffffff"

		clear()
		addLine(string.format(_INFM('InfoHeader'), color, itemInfo.itemName))

		local sell = itemInfo.itemSell or itemInfo.sell or 0
		local count = itemInfo.itemCount or 1

		if (sell > 0) then
			local sgsc = EnhTooltip.GetTextGSC(sell, true)

			if (count and (count > 1)) then
				local sqgsc = EnhTooltip.GetTextGSC(sell*count, true)
				addLine(string.format(_INFM('FrmtInfoSellmult'), count, sgsc)..": "..sqgsc, "ee8822")
			else
				addLine(string.format(_INFM('FrmtInfoSell'))..": "..sgsc, "ee8822")
			end
		end

		if (itemInfo.stack > 1) then
			addLine(string.format(_INFM('FrmtInfoStx'), itemInfo.stack))
		end

		local reagentInfo = ""
		if (itemInfo.classText) then
			reagentInfo = string.format(_INFM('FrmtInfoClass'), itemInfo.classText)
			addLine(reagentInfo, "aa66ee")
		end
		if (itemInfo.usageText) then
			reagentInfo = string.format(_INFM('FrmtInfoUse'), itemInfo.usageText)
			addLine(reagentInfo, "aa66ee")
		end

		if (itemInfo.isPlayerMade) then
			addLine(string.format(_INFM('InfoPlayerMade'), itemInfo.reqLevel, itemInfo.reqSkillName), "5060ff")
		end
		
		InformantFrame:Show()
	else
		clear()
		addLine(_INFM('InfoNoItem'), "ff4010")
		InformantFrame:Show()
	end
end

function onQuit()
	if (not InformantConfig.position) then
		InformantConfig.position = { }
	end
	InformantConfig.position.x, InformantConfig.position.y = InformantFrame:GetCenter()
end

function onLoad()
	this:RegisterEvent("ADDON_LOADED")

	if (not InformantConfig) then
		InformantConfig = {}
		setFilterDefaults()
	end

	InformantFrameTitle:SetText(_INFM('FrameTitle'))
end

local function frameLoaded()
	Stubby.RegisterFunctionHook("EnhTooltip.AddTooltip", 300, tooltipHandler)

	onLoad()
end

function onVariablesLoaded()
	setFilterDefaults()

	InformantFrameTitle:SetText(_INFM('FrameTitle'))

	if (InformantConfig.position) then
		InformantFrame:ClearAllPoints()
		InformantFrame:SetPoint("CENTER", UIParent, "BOTTOMLEFT", InformantConfig.position.x, InformantConfig.position.y)
	end

	-- Restore key bindings
	-- This workaround is required for LoadOnDemand addons since their saved
	-- bindings are deleted upon login.
	local profile = getKeyBindProfile();
	if (InformantConfig and InformantConfig.bindings) then
		if (not	InformantConfig.bindings[profile]) then profile = 'global'; end
		if (InformantConfig.bindings[profile]) then
			for _,key in ipairs(InformantConfig.bindings[profile]) do
				SetBinding(key, 'INFORMANT_POPUPDOWN')
			end
		end
	end
	this:RegisterEvent("UPDATE_BINDINGS")	-- Monitor changes to bindings

	Informant.InitCommands()
end

function onEvent(event)
	if (event == "ADDON_LOADED" and string.lower(arg1) == "informant") then
		onVariablesLoaded()
		this:UnregisterEvent("ADDON_LOADED")
	elseif (event == "UPDATE_BINDINGS") then
		-- Store key bindings for Informant
		local key1, key2 = GetBindingKey('INFORMANT_POPUPDOWN');
		local profile = getKeyBindProfile();

		if (not InformantConfig.bindings) then InformantConfig.bindings = {}; end
		if (not InformantConfig.bindings[profile]) then InformantConfig.bindings[profile] = {}; end

		InformantConfig.bindings[profile][1] = key1;
		InformantConfig.bindings[profile][2] = key2;
	end
end

function frameActive(isActive)
	if (isActive) then
		scrollUpdate(0)
	end
end

function getRowCount()
	return table.getn(lines)
end

function scrollUpdate(offset)
	local numLines = getRowCount()
	if (numLines > 25) then
		if (not offset) then
			offset = FauxScrollFrame_GetOffset(InformantFrameScrollBar)
		else
			if (offset > numLines - 25) then offset = numLines - 25 end
			FauxScrollFrame_SetOffset(InformantFrameScrollBar, offset)
		end
	else
		offset = 0
	end
	local line
	for i=1, 25 do
		line = lines[i+offset]
		local f = getglobal("InformantFrameText"..i)
		if (line) then
			f:SetText(line)
			f:Show()
		else
			f:Hide()
		end
	end
	if (numLines > 25) then
		FauxScrollFrame_Update(InformantFrameScrollBar, numLines, 25, numLines)
		InformantFrameScrollBar:Show()
	else
		InformantFrameScrollBar:Hide()
	end
end

function testWrap(text)
	InformantFrameTextTest:SetText(text)
	if (InformantFrameTextTest:GetWidth() < InformantFrame:GetWidth() - 20) then
		return text, ""
	end

	local pos, test, best, rest
	best = text
	rest = nil
	pos = string.find(text, " ")
	while (pos) do
		test = string.sub(text, 1, pos-1)
		InformantFrameTextTest:SetText(test)
		if (InformantFrameTextTest:GetWidth() < InformantFrame:GetWidth() - 20) or (not rest) then
			best = test
			rest = string.sub(test, pos+1)
		else
			break
		end
		pos = string.find(text, " ", pos+1)
	end
	return best, rest
end

function addLine(text, color, level)
	if (text == nil) then return end
	if (not level) then level = 1 end
	if (level > 100) then
		return
	end

	if (type(text) == "table") then
		for pos, line in text do
			addLine(line, color, level)
		end
		return
	end

	if (not text) then
		table.insert(lines, "nil")
	else
		local best, rest = testWrap(text)
		if (color) then
			table.insert(lines, string.format("|cff%s%s|r", color, best))
		else
			table.insert(lines, best)
		end
		if (rest) and (rest ~= "") then
			addLine(rest, color, level+1)
		end
	end
	scrollUpdate()
end

function clear()
	lines = {}
	scrollUpdate()
end

function setFilterDefaults()
	if (not InformantConfig.filters) then InformantConfig.filters = {}; end
	for k,v in pairs(filterDefaults) do
		if (InformantConfig.filters[k] == nil) then
			InformantConfig.filters[k] = v;
		end
	end
end

-- Key binding helper functions

function getKeyBindProfile()
	if (IsAddOnLoaded("PerCharBinding")) then
		return GetRealmName() .. ":" .. UnitName("player")
	end
	return 'global'
end

-- GLOBAL OBJECT

Informant = {
	version = INFORMANT_VERSION,
	GetItem = getItem,
	GetRowCount = getRowCount,
	AddLine = addLine,
	Clear = clear,
	ShowHideInfo = showHideInfo,
	SetVendors = setVendors,
	SetDatabase = setDatabase,
	FrameActive = frameActive,
	FrameLoaded = frameLoaded,
	ScrollUpdate = scrollUpdate,
	GetFilter = getFilter,
	GetFilterVal = getFilterVal,
	GetLocale = getLocale,
	OnEvent = onEvent,
	SetFilter = setFilter,
	SetFilterDefaults = setFilterDefaults,
}

