MobHealth3 = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceConsole-2.0")
local MHC = LibStub("LibMobHealthCache-1.0")

--[[
	File-scope local vars
--]]
if MHC.Health_Cache then MH3Cache = MHC.Health_Cache else MH3Cache = {} end

local AccumulatorHP = {} -- Keeps Damage-taken data for mobs that we've actually poked during this session
local AccumulatorPerc = {} -- Keeps Percentage-taken data for mobs that we've actually poked during this session
local calculationUnneeded = {} -- Keeps a list of things that don't need calculation (e.g. Beast Lore'd mobs)

local currentAccHP
local currentAccPerc

local targetName, targetLevel, targetIndex
local recentDamage, totalDamage = 0, 0
local startPercent, lastPercent = 100, 100

local defaults = {
    saveData = true,
    precision = 10,
    stableMax = true,
}

-- Metatable that provides compat for mods that index MobHealthDB directly
local compatMT ={
    __index = function(t, k)
        return MH3Cache[k] and MH3Cache[k].."/100";
    end
}

-- Debug function. Not for Joe Average
function GetMH3Cache() return MH3Cache end


--[[
	Init/Enable methods
--]]

function MobHealth3:OnInitialize()
	-- If config savedvars don't exist, create them
	MobHealth3Config = MobHealth3Config or defaults

	-- If the user is saving data, then load it into the cache
	if MobHealth3DB and MobHealth3Config.saveData then
		MH3Cache = MobHealth3DB
	elseif (MobHealth3Config.saveData) then
		MobHealth3DB = MH3Cache
	end

	self:RegisterChatCommand({"/mobhealth3", "/mh3"}, {
		type = "group",
		args = {
			save = {
				name = "数据保存",
				desc = "保存敌人血量数据。",
				type = "toggle",
				get = function() return not not MobHealth3DB end, -- "Double negatives for the not lose!" -Wobin
				set = function(val)
					if val == false then
						MobHealth3DB = nil
						MobHealth3Config.saveData = val
					else
						MobHealth3DB = MH3Cache
						MobHealth3Config.saveData = val
					end
				end,
			},
			precision = {
				name = "精度估算",
				desc = "该数值越低，则目标血量值变动越快，血量变化的估算精度就会越不准确。（默认值为10）",
				type = "range",
				step = 1,
				min = 1,
				max = 99,
				get = function() return MobHealth3Config.precision end,
				set = function(val) MobHealth3Config.precision = tonumber(val) end,
			},
            stablemax = {
                name = "稳定最大生命值",
                desc = "勾选此项，你的目标最大生命值变化只更新一次。如果目标数据是未知的，MH3将会在最近的一次战斗中取准确值。",
                type = "toggle",
                get = function() return MobHealth3Config.stableMax end,
                set = function(val) MobHealth3Config.stableMax = val end,
            },
            reset = {
                name = "重置缓存/数据库",
                desc = "重置缓存/数据库到你保存的设置值。",
                type = "execute",
                func = function()
                    MH3Cache = {}
                    AccumulatorHP = {}
                    AccumulatorPerc = {}
                    MobHealth3DB = MobHealth3Config.saveData and {} or nil
                    self:Print("Cache/Database reset")
                end,
            },
		},
	})

    -- MH/MH2 database converter. MI2 too if the user follows the instructions
    if MobHealthDB and not MobHealthDB.thisIsADummyTable then
        -- Turn on saving
        MobHealth3DB = MH3Cache
        MobHealth3Config.saveData = true

        for k,v in pairs(MobHealthDB) do
            local _, _, pts, pct = string.find(v, "(%d+)/(%d+)")
            if pts then
                local maxHP = math.floor(pts/pct * 100 + 0.5)
                MH3Cache[k] = maxHP
            end
        end
        self:Print("老mobhealth / mobhealth2 / mobinfo2数据库检测和转换。MH3也自动打开你保存的数据存储。")
    end

    MobHealthDB = { thisIsADummyTable = true }
    setmetatable(MobHealthDB, compatMT) -- Metamethod proxy ENGAGE!! </cheesiness>
end

function MobHealth3:OnEnable()
	self:RegisterEvent("UNIT_COMBAT")
	self:RegisterEvent("PLAYER_TARGET_CHANGED")
	self:RegisterEvent("UNIT_HEALTH")
end

--[[
	Dummy MobHealthFrame. Some mods use this to detect MH/MH2/MI2
--]]

CreateFrame("frame", "MobHealthFrame")

--[[
    Event Handlers
--]]

function MobHealth3:UNIT_COMBAT()
	if arg1=="target" and currentAccHP then
		recentDamage = recentDamage + arg4
		totalDamage = totalDamage + arg4
	end
end

function MobHealth3:PLAYER_TARGET_CHANGED()
    if MobHealth3Config.stableMax and currentAccHP and currentAccHP > 0 and currentAccPerc > 0 then
        -- Save now if we have actual values (0 /0 --> 1.#IND == BAD)
        MH3Cache[targetIndex] = math.floor( currentAccHP / currentAccPerc * 100 + 0.5 )
    end

	-- Is target valid?
	-- We ignore pets. There's simply far too many pets that share names with players so we let players take priority

	local creatureType = UnitCreatureType("target") -- Saves us from calling it twice
	if UnitCanAttack("player", "target") and not UnitIsDead("target") and not UnitIsFriend("player", "target") and not ( (creatureType == MOBHEALTH_UnitCreatureType_Beast or creatureType == MOBHEALTH_UnitCreatureType_Demon) and UnitPlayerControlled("target") ) then

		targetName = UnitName("target")
		targetLevel = UnitLevel("target")

		targetIndex = string.format("%s:%d", targetName, targetLevel)

		--self:Debug("Acquired valid target: index: %s, in db: %s", targetIndex, not not MH3Cache[targetIndex])

		recentDamage, totalDamage = 0, 0, 0
		startPercent = UnitHealth("target")
		lastPercent = startPercent

		currentAccHP = AccumulatorHP[targetIndex]
		currentAccPerc = AccumulatorPerc[targetIndex]

		if not UnitIsPlayer("target") then
			-- Mob: keep accumulated percentage below 200% in case we hit mobs with different hp
			if not currentAccHP then
				if MH3Cache[targetIndex] then
					-- We claim that this previous value that we have is from seeing percentage drop from 100 to 0
					AccumulatorHP[targetIndex] = MH3Cache[targetIndex]
					AccumulatorPerc[targetIndex] = 100
				else
					-- Nothing previously known. Start fresh.
					AccumulatorHP[targetIndex] = 0
					AccumulatorPerc[targetIndex] = 0
				end
				currentAccHP = AccumulatorHP[targetIndex]
				currentAccPerc = AccumulatorPerc[targetIndex]
			end

			if currentAccPerc>200 then
				currentAccHP = currentAccHP / currentAccPerc * 100
				currentAccPerc = 100
			end

		else
			-- Player health can change a lot. Different gear, buffs, etc.. we only assume that we've seen 10% knocked off players previously
			if not currentAccHP then
				if MH3Cache[targetIndex] then
					AccumulatorHP[targetIndex] = MH3Cache[targetIndex]*0.1
					AccumulatorPerc[targetIndex] = 10
				else
					AccumulatorHP[targetIndex] = 0
					AccumulatorPerc[targetIndex] = 0
				end
				currentAccHP = AccumulatorHP[targetIndex]
				currentAccPerc = AccumulatorPerc[targetIndex]
			end

			if currentAccPerc>10 then
				currentAccHP = currentAccHP / currentAccPerc * 10
				currentAccPerc = 10
			end

		end

	else
		--self:Debug("Acquired invalid target. Ignoring")
		currentAccHP = nil
		currentAccPerc = nil
	end
end

function MobHealth3:UNIT_HEALTH()
	if currentAccHP and arg1=="target" then
		self:CalculateMaxHealth(UnitHealth("target"), UnitHealthMax("target"))
	end
end

--[[
	The meat of the machine!
--]]

function MobHealth3:CalculateMaxHealth(current, max)

	if calculationUnneeded[targetIndex] then return;

    elseif current==startPercent or current==0 then
	--self:Debug("Targetting a dead guy?")

    elseif max > 100 then
        -- zOMG! Beast Lore! We no need no stinking calculations!
        MH3Cache[targetIndex] = max
        -- print(string.format("We got beast lore! Max is %d", max))
        calculationUnneeded[targetIndex] = true

	elseif current > lastPercent or startPercent>100 then
		-- Oh noes! It healed! :O
		lastPercent = current
		startPercent = current
		recentDamage=0
		totalDamage=0
		--self:Debug("O NOES IT HEALED!?")

	elseif recentDamage>0 then

		if current~=lastPercent then
			currentAccHP = currentAccHP + recentDamage
			currentAccPerc = currentAccPerc + (lastPercent-current)
			recentDamage = 0
			lastPercent = current

			if currentAccPerc >= MobHealth3Config.precision and not (MobHealth3Config.stableMax and MH3Cache[targetIndex]) then
				MH3Cache[targetIndex] = math.floor( currentAccHP / currentAccPerc * 100 + 0.5 )
				--self:Debug("Caching %s as %d", targetIndex, MH3Cache[targetIndex])
			end

		end

	end
end

--[[
	Compatibility for functions MobHealth2 introduced
--]]

function MobHealth_GetTargetMaxHP()
	local currHP, maxHP, found = MobHealth3:GetUnitHealth("target", UnitHealth("target"), UnitHealthMax("target"), UnitName("target"), UnitLevel("target"))
	return found and maxHP or nil
end

function MobHealth_GetTargetCurHP()
	local currHP, maxHP, found = MobHealth3:GetUnitHealth("target", UnitHealth("target"), UnitHealthMax("target"), UnitName("target"), UnitLevel("target"))
	return found and currHP or nil
end

--[[
	Compatibility for MobHealth_PPP()
--]]

function MobHealth_PPP(index)
	return MH3Cache[index] and MH3Cache[index]/100 or 0
end

--[[
	MobHealth3 API

	I'm using MediaWiki formatting for the docs here so I can easily copy/paste if I make modifications
--]]

--[[
== MobHealth3:GetUnitHealth(unit[,current][, max][, name][, level]) ==
Returns the current and max health of the specified unit

=== Args ===
; unit : The unitID you want health for
; [current] : (optional) The value of UnitHealth(unit). Passing this if your mod knows it saves MH3 from having to call UnitHealth itself<br>
; [max] : (optional) The value of UnitHealthMax(unit). Passing this if your mod knows it saves MH3 from having to call UnitHealthMax itself<br>
; [name] : (optional) The name of the unit. Passing this if your mod knows it saves MH3 from having to call UnitName itself<br>
; [level] : (optional) The level of the unit. Passing this if your mod knows it saves MH3 from having to call UnitLevel itself

=== Returns ===
If the specified unit is alive, hostile and its true max health unknown, the absolute current and max value based on the cache's current entry<br>
If the specified unit's friendly or data was not found, returns UnitHealth(unit) and UnitHealthMax(unit)

A third return value is a boolean stating whether an estimation was found

=== Remarks ===
Remember, args 2-5 are optional and passing the args if your mod already knows them saves MH3 from having to find the data itself<br>
Don't pass level as a string. Please.<br>
MobHealth3 does the target-validty checking for you

=== Example ===
 function YourUnitFrames:UpdateTargetFrame()
 	local name, level = UnitName("target"), UnitLevel("target")
 	local cur, max, found
 	if MobHealth3 then
 		cur, max, found = MobHealth3:GetUnitHealth("target", UnitHealth("target"), UnitHealthMax("target"), name, level)
 	else
 		cur, max = UnitHealth("target"), UnitHealthMax("target")
 	end
 	YourTargetFrame.HealthText:SetText(cur .. "/" .. max)
 	YourTargetFrame.NameText:SetText(name)
 	YourTargetFrame.LevelText:SetText(level)
 end
--]]
function MobHealth3:GetUnitHealth(unit, current, max, name, level)
	if not UnitExists(unit) then return 0, 0, false; end

	current = current or UnitHealth(unit)
	max = max or UnitHealthMax(unit)
	name = name or UnitName(unit)
	level = level or UnitLevel(unit)

	-- Mini validity check.
	-- No need to do the full thing because indexing the cache and getting nil back is much faster.
	-- Remember, an invalid target should never be in the cache
	-- The only parts we actually have to do are:
        -- Pet check
        -- Beast Lore check (Does UnitHealthMax give us the real value?)

	local creatureType = UnitCreatureType(unit) -- Saves us from calling it twice
	if max == 100 and not ( (creatureType == MOBHEALTH_UnitCreatureType_Demon or creatureType == MOBHEALTH_UnitCreatureType_Beast) and UnitPlayerControlled(unit) ) then
		local maxHP = MH3Cache[string.format("%s:%d", name, level)]

		if maxHP then return math.floor(current/100 * maxHP + 0.5), maxHP, true; end
	end

	-- If not maxHP or we're dealing with an invalid target
	return current, max, false;
end